<?php
session_start();
require_once '../config/db.php';
header('Content-Type: application/json');

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    echo json_encode(['success' => false, 'message' => 'Not authorized']);
    exit();
}

// Get M-Pesa settings
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'mpesa_settings'");
$stmt->execute();
$mpesaSettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);

// Check if M-Pesa is enabled and all required credentials are set
if (($mpesaSettings['enable_mpesa'] ?? '0') !== '1') {
    echo json_encode(['success' => false, 'message' => 'M-Pesa payments are currently disabled.']);
    exit();
}

// Validate required credentials
$requiredCredentials = ['mpesa_consumer_key', 'mpesa_consumer_secret', 'mpesa_shortcode', 'mpesa_passkey'];
foreach ($requiredCredentials as $credential) {
    if (empty($mpesaSettings[$credential])) {
        echo json_encode(['success' => false, 'message' => 'M-Pesa API credentials are not properly configured.']);
        exit();
    }
}

// Get POST data
$data = json_decode(file_get_contents('php://input'), true);
$amount = floatval($data['amount'] ?? 0);
$phone = preg_replace('/\D/', '', $data['phone'] ?? '');
$customerId = $data['customer_id'] ?? null;
$customerName = $data['customer_name'] ?? '';

if ($amount <= 0 || strlen($phone) < 10) {
    echo json_encode(['success' => false, 'message' => 'Invalid amount or phone number.']);
    exit();
}

// Format phone number to 254 format if not already
if (!preg_match('/^254/', $phone)) {
    $phone = '254' . ltrim($phone, '0');
}

// Daraja API credentials
$consumerKey = $mpesaSettings['mpesa_consumer_key'];
$consumerSecret = $mpesaSettings['mpesa_consumer_secret'];
$shortcode = $mpesaSettings['mpesa_shortcode'];
$passkey = $mpesaSettings['mpesa_passkey'];
$env = $mpesaSettings['mpesa_env'] ?? 'sandbox';

// Daraja API URLs
$baseUrl = ($env === 'production') ? 'https://api.safaricom.co.ke' : 'https://sandbox.safaricom.co.ke';

try {
    // 1. Get access token
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $baseUrl . '/oauth/v1/generate?grant_type=client_credentials');
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Authorization: Basic ' . base64_encode($consumerKey . ':' . $consumerSecret)]);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    
    if (curl_errno($ch)) {
        $curlError = curl_error($ch);
        curl_close($ch);
        throw new Exception('Failed to get access token: ' . $curlError);
    }
    
    curl_close($ch);
    
    // Log the response for debugging
    error_log('M-Pesa Access Token Response: ' . $response);
    error_log('M-Pesa HTTP Code: ' . $httpCode);
    
    $result = json_decode($response, true);
    
    // Check if response is valid JSON
    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception('Invalid JSON response from M-Pesa API: ' . json_last_error_msg());
    }
    
    // Check for M-Pesa error response
    if (isset($result['errorCode'])) {
        throw new Exception('M-Pesa API Error: ' . ($result['errorMessage'] ?? 'Unknown error') . ' (Code: ' . $result['errorCode'] . ')');
    }
    
    // Check for access token
    if (!isset($result['access_token'])) {
        $errorDetails = isset($result['error_description']) ? $result['error_description'] : 'No access token in response';
        throw new Exception('Invalid access token response: ' . $errorDetails . '. Full response: ' . $response);
    }
    
    $accessToken = $result['access_token'];

    // 2. Initiate STK Push
    $timestamp = date('YmdHis');
    $password = base64_encode($shortcode . $passkey . $timestamp);
    
    $stkPushData = [
        'BusinessShortCode' => $shortcode,
        'Password' => $password,
        'Timestamp' => $timestamp,
        'TransactionType' => 'CustomerPayBillOnline',
        'Amount' => $amount,
        'PartyA' => $phone,
        'PartyB' => $shortcode,
        'PhoneNumber' => $phone,
        'CallBackURL' => 'https://' . $_SERVER['HTTP_HOST'] . '/api/mpesa_callback.php',
        'AccountReference' => $customerId ? "CUST{$customerId}" : 'POS',
        'TransactionDesc' => $customerName ? "Payment by {$customerName}" : 'POS Payment'
    ];

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $baseUrl . '/mpesa/stkpush/v1/processrequest');
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Authorization: Bearer ' . $accessToken,
        'Content-Type: application/json'
    ]);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($stkPushData));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
    
    $response = curl_exec($ch);
    
    if (curl_errno($ch)) {
        throw new Exception('Failed to initiate STK Push: ' . curl_error($ch));
    }
    
    $result = json_decode($response, true);
    curl_close($ch);

    if (isset($result['ResponseCode']) && $result['ResponseCode'] === '0') {
        // Log the STK Push request
        $stmt = $pdo->prepare("INSERT INTO mpesa_transactions (phone, amount, customer_id, customer_name, merchant_request_id, checkout_request_id, response_code, response_description, created_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW())");
        $stmt->execute([
            $phone,
            $amount,
            $customerId,
            $customerName,
            $result['MerchantRequestID'] ?? null,
            $result['CheckoutRequestID'] ?? null,
            $result['ResponseCode'] ?? null,
            $result['ResponseDescription'] ?? null
        ]);

        // Create a pending credit payment record if customer_id is provided
        if ($customerId) {
            // Get the oldest unpaid invoices for this customer
            $stmt = $pdo->prepare("
                SELECT 
                    ct.id as credit_transaction_id,
                    ct.sale_id,
                    ct.amount,
                    COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) as paid,
                    ct.amount - COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) as balance
                FROM credit_transactions ct
                LEFT JOIN credit_payments cp ON ct.id = cp.credit_transaction_id
                WHERE ct.customer_id = ?
                GROUP BY ct.id
                HAVING balance > 0
                ORDER BY ct.created_at ASC
            ");
            $stmt->execute([$customerId]);
            $unpaidInvoices = $stmt->fetchAll();
            
            if (!empty($unpaidInvoices)) {
                $remainingAmount = $amount;
                
                // Create credit payment records for each invoice (up to the payment amount)
                foreach ($unpaidInvoices as $invoice) {
                    if ($remainingAmount <= 0) break;
                    
                    $paymentAmount = min($remainingAmount, $invoice['balance']);
                    
                    // Insert pending credit payment record
                    $stmt = $pdo->prepare("
                        INSERT INTO credit_payments 
                        (credit_transaction_id, sale_id, customer_id, amount, payment_method, reference, checkout_request_id, status, created_by, created_at) 
                        VALUES (?, ?, ?, ?, 'mpesa', ?, ?, 'pending', ?, NOW())
                    ");
                    
                    $stmt->execute([
                        $invoice['credit_transaction_id'],
                        $invoice['sale_id'],
                        $customerId,
                        $paymentAmount,
                        'MPESA-' . ($result['CheckoutRequestID'] ?? ''),
                        $result['CheckoutRequestID'] ?? null,
                        $_SESSION['user_id'] ?? null
                    ]);
                    
                    $remainingAmount -= $paymentAmount;
                }
            }
        }

        echo json_encode(['success' => true, 'message' => 'STK Push initiated successfully']);
    } else {
        throw new Exception($result['ResponseDescription'] ?? 'Failed to initiate STK Push');
    }
} catch (Exception $e) {
    error_log('M-Pesa Error: ' . $e->getMessage());
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
} 